/**********************************************************************************/
/* MyFirstDeviceAPI.cpp													*/
/*																		*/
/* Bruno van Dooren														*/
/* 																		*/
/* a project to show the usage of a device driver, and building a class based api.	*/
/* connecting to a device driver is mostly done via a shared library. it is just	*/
/* as possible to connect an aplication directly to a device driver, but there are	*/
/* 2 important reasons why you should do this:								*/
/* -if every application has to connect and communicate with a device driver,		*/
/*  chances are that at least one application will contain a serious bug, and		*/
/*  during execution will crash the device driver, and maybe even the system.		*/
/* -creating a class based api will hide a lot of the low level bits, flags and	*/
/*  pointers from the application. this makes the api easier to use, and the		*/
/*  of bugs smaller, since it is easier to maintain 1 api than x applications		*/
/**********************************************************************************/

#include <SupportDefs.h>
#include <Errors.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include "MyFirstDeviceAPI.h"
#include "MyFirstDeviceDriver.h"

/**********************************************************************************/
/* construction and destruction of the device api object						*/
/**********************************************************************************/
MyFirstDeviceAPI::MyFirstDeviceAPI()
{
	DevConnection = -1;
	
	//open the device fo rwhich this API was made
	DevConnection = open("/dev/misc/MyFirstDeviceDriver", O_RDWR);
	if (DevConnection < 0) {
		throw("Error opening device");
	}
}

MyFirstDeviceAPI::~MyFirstDeviceAPI()
{
	close(DevConnection);
}

/**********************************************************************************/
/* read a range of ints from the register range on the device					*/
/**********************************************************************************/
void MyFirstDeviceAPI::ReadRange(int32* IntArray, int32 Offset, int32* NumElements)
{
	/*read a range of integers from the register range. throw an exception
	if the read operation fails*/
	
	/*transform the number of ints to the number of bytes*/
	size_t NumBytes = *NumElements * sizeof(int32);
	if(B_NO_ERROR /*!=*/> read_pos(DevConnection,
								Offset,
								(unsigned char *)IntArray,
								NumBytes))
		throw("Read operation failed");
}

/**********************************************************************************/
/* write a range of ints to the register range on the device					*/
/**********************************************************************************/
void MyFirstDeviceAPI::WriteRange(const int32* IntArray, int32 Offset, int32* NumElements)
{
	/*write a range of integers to the register range. throw an exception
	if the read operation fails*/
	size_t NumBytes = *NumElements * sizeof(int32);
	
	if(B_NO_ERROR /*!=*/> write_pos(DevConnection,
								Offset,
								(unsigned char *)IntArray,
								NumBytes))
		throw("Write operation failed");
}

/**********************************************************************************/
/* get the number of operations that have been executed with this connection		*/
/**********************************************************************************/
int32 MyFirstDeviceAPI::GetNumOps(void)
{
	/*get the number of operations that have been executed with this connection.
	throw an exception if the read operation fails*/
	int32 NumOps = -1;
	if(B_NO_ERROR != ioctl(DevConnection,
							op_GETNUMOPS,
							&NumOps,
							sizeof(NumOps)))
		throw("op: 'GetNumOps' failed");
	
	return NumOps;
}

/**********************************************************************************/
/* clear the complete register range.										*/
/**********************************************************************************/
void MyFirstDeviceAPI::ClearAllRegs(void)
{
	/*clear all registers on the device. throw an exception
	if the read operation fails*/
	if(B_NO_ERROR != ioctl(DevConnection, op_CLEARALL, 0, 0))
		throw("op: 'ClearAll' failed");
}
